<?php
/* --------------------------------------------------------------
   PermissionsReader.php 2020-04-21
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Permission\Repositories;

use Doctrine\DBAL\Connection;
use Gambio\Core\Permission\PermissionDefinition;
use Gambio\Core\Permission\UserId;

/**
 * Class PermissionsReader
 *
 * @package Gambio\Core\Auth\Repositories
 */
class PermissionsReader
{
    /**
     * @var Connection
     */
    private $db;
    
    
    /**
     * PermissionsReader constructor.
     *
     * @param Connection $db
     */
    public function __construct(Connection $db)
    {
        $this->db = $db;
    }
    
    
    /**
     * Returns the state of the requested permission.
     *
     * @param UserId               $id
     * @param PermissionDefinition $permissionDefinition
     *
     * @return bool
     */
    public function isPermissionGranted(UserId $id, PermissionDefinition $permissionDefinition): bool
    {
        if ($this->adminAccessGroupExists($permissionDefinition)) {
            $permissions = $this->getPermissionsForKnownItems($id, $permissionDefinition);
        } else {
            $permissions = $this->getPermissionsForUnknownItems($id);
        }
        
        switch ($permissionDefinition->action()) {
            case 'read':
                return $permissions['reading'];
            case 'write':
                return $permissions['writing'];
            case 'delete':
                return $permissions['deleting'];
        }
    }
    
    
    /**
     * @param UserId               $id
     * @param PermissionDefinition $permissionDefinition
     *
     * @return bool[]
     */
    private function getPermissionsForKnownItems(
        UserId $id,
        PermissionDefinition $permissionDefinition
    ): array {
        $permissions = [
            'reading'  => false,
            'writing'  => false,
            'deleting' => false,
        ];
        
        $result = $this->db->createQueryBuilder()
            ->select('admin_access_permissions.*')
            ->from('admin_access_permissions')
            ->join('admin_access_permissions',
                   'admin_access_group_items',
                   'admin_access_group_items',
                   'admin_access_permissions.admin_access_group_id = admin_access_group_items.admin_access_group_id')
            ->join('admin_access_permissions',
                   'admin_access_users',
                   'admin_access_users',
                   'admin_access_permissions.admin_access_role_id = admin_access_users.admin_access_role_id')
            ->where('admin_access_users.customer_id = :id AND admin_access_group_items.identifier = :identifier AND admin_access_group_items.type = :type')
            ->setParameter('identifier', $permissionDefinition->identifier())
            ->setParameter('type', strtoupper($permissionDefinition->type()))
            ->setParameter('id', $id->userId())
            ->execute()
            ->fetch();
        
        if ($result !== false) {
            $permissions = [
                'reading'  => $result['reading_granted'] === '1',
                'writing'  => $result['writing_granted'] === '1',
                'deleting' => $result['deleting_granted'] === '1',
            ];
        }
        
        return $permissions;
    }
    
    
    /**
     * @param UserId $id
     *
     * @return bool[]
     */
    private function getPermissionsForUnknownItems(UserId $id): array
    {
        $permissions = [
            'reading'  => false,
            'writing'  => false,
            'deleting' => false,
        ];
        
        $query = $this->db->createQueryBuilder()
            ->select('*')
            ->from('admin_access_roles')
            ->join('admin_access_roles',
                   'admin_access_users',
                   'admin_access_users',
                   'admin_access_roles.admin_access_role_id = admin_access_users.admin_access_role_id')
            ->where('admin_access_users.customer_id = :userId')
            ->setParameter('userId', $id->userId());
        
        $rolePermissions = $query->execute()->fetchAll();
        if (count($rolePermissions) > 0) {
            foreach ($rolePermissions as $permission) {
                if ($permission['reading_unknown_group_granted'] === '1') {
                    $permissions['reading'] = true;
                }
                if ($permission['writing_unknown_group_granted'] === '1') {
                    $permissions['writing'] = true;
                }
                if ($permission['deleting_unknown_group_granted'] === '1') {
                    $permissions['deleting'] = true;
                }
            }
        }
        
        return $permissions;
    }
    
    
    /**
     * @param PermissionDefinition $permissionDefinition
     *
     * @return bool
     */
    private function adminAccessGroupExists(PermissionDefinition $permissionDefinition): bool
    {
        $groupData = $this->db->createQueryBuilder()
            ->select('admin_access_group_id')
            ->from('admin_access_group_items')
            ->where('identifier = :identifier AND type = :type')
            ->setParameter('identifier', $permissionDefinition->identifier())
            ->setParameter('type', $permissionDefinition->type())
            ->execute()
            ->fetch();
        
        return $groupData !== false;
    }
}